<?php

namespace App\Http\Controllers\AcademicManager;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\DoubtSession;
use App\Models\Account;
use App\Models\LmsCourse;
use App\Models\LmsClasses;

class DoubtSessionsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $doubt_sessions = DoubtSession::leftJoin('lms_courses', 'lms_courses.id', 'doubt_sessions.course_id')
            ->leftJoin('lms_classes', 'lms_classes.id', 'doubt_sessions.class_id')
            ->leftJoin('accounts', 'accounts.id', 'doubt_sessions.student_id')
            ->leftJoin('accounts as teacher', 'teacher.id', 'doubt_sessions.teacher_id')
            ->select(
                'doubt_sessions.*',
                'lms_courses.course_title as course' ,
                'lms_classes.class_name as class' ,
                'accounts.name as student' ,
        );

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype', 'ASC'); 
            $search_query = $request->get('query');

            $doubt_sessions = $doubt_sessions->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('accounts.name', 'like', '%'.$search_query.'%')
                        ->orWhere('accounts.name', 'like', '%'.$search_query.'%')
                        ->orWhere('lms_courses.course_title', 'like', '%'.$search_query.'%')
                        ->orWhere('lms_classes.class_name', 'like', '%'.$search_query.'%')
                        ->orWhere('doubt_sessions.subject', 'like', '%'.$search_query.'%')
                        ->orWhere('doubt_sessions.description', 'like', '%'.$search_query.'%')
                        ->orWhere('doubt_sessions.scheduled_date', 'like', '%'.$search_query.'%');
            })
            ->orderBy('doubt_sessions.id', $sort_type)
            ->paginate($sort_by);

            return view('doubt-sessions.table', compact('doubt_sessions'));
        } else {
            $doubt_sessions = $doubt_sessions->orderBy('doubt_sessions.id', 'ASC')
            ->paginate(10);

            return view('doubt-sessions.index', compact('doubt_sessions'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        $classes = LmsClasses::all();
        $courses = LmsCourse::all();

        return view('doubt-sessions.create' ,compact('courses','classes'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'user_id'        => 'required',
            'course_id'      => 'required',
            'class_id'       => 'required',
            'subject'        => 'required',
            'description'    => 'required',
            'scheduled_date' => 'required',
        ]);   


        if ($validator->fails()) {

            return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                'redirect' => 'doubt-sessions.index'],
            ]);
        }

        $student = $request->user_id;

        $doubt_sessions = DoubtSession::create([
            'student_id'     => $student,
            'course_id'      => $request->course_id,
            'class_id'       => $request->class_id,
            'subject'        => $request->subject,
            'description'    => $request->description,
            'scheduled_date' => $request->scheduled_date,
        ]);

        return response()->json([
            'success'  => true,
            'message'  => 'Doubt Session created successfully',
            'data'     => [
            'redirect' => 'doubt-sessions.index', 
            ]
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $doubt_sessions = DoubtSession::findOrFail($id);
        $accounts       = Account::select('id' , 'name')
        ->where('id' , $doubt_sessions->student_id)
        ->first();

        $courses = LmsCourse::all();
        $classes = LmsClasses::all();

        return view('doubt-sessions.edit', compact('doubt_sessions' ,'accounts' , 'courses' ,'classes'));

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'user_id'        => 'required',
            'course_id'      => 'required',
            'class_id'       => 'required',
            'subject'        => 'required',
            'description'    => 'required',
            'scheduled_date' => 'required',
        ]);

        if ($validator->fails()) {
           return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                'redirect' => 'doubt-sessions.index'],
            ]);
        }

        $doubt_sessions = DoubtSession::findOrFail($id);
        $student        = $request->user_id;
        $doubt_sessions->update([
            'student_id'     => $student,
            'course_id'      => $request->course_id,
            'class_id'       => $request->class_id,
            'subject'        => $request->subject,
            'description'    => $request->description,
            'scheduled_date' => $request->scheduled_date,
        ]);

        return response()->json([
            'success'  => true,
            'message'  => 'Doubt Session updated successfully',
            'data'     => [
                'redirect' => 'doubt-sessions.index', 
            ]
        ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $doubt_sessions = DoubtSession::findOrFail($id);
        $doubt_sessions->delete();

        return response()->json([
            'success' => true,
            'message' => 'Doubt Session Deleted successfully',
            'data'    => [
                'redirect' => '', 
            ]
        ]);
    }
}
