<?php

namespace App\Http\Controllers\Accounting;

use App\Exports\FinancialAccountVoucher\JournalVoucherExport;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CompanyAddress;
use App\Models\Account;
use App\Models\ProductBrand;
use App\Models\Product;
use App\Models\PaymentMode;
use App\Models\SaleInvoice;
use App\Models\SaleInvoiceType;
use App\Models\SaleInvoiceDetail;
use App\Models\ProductPrice;
use App\Models\ProductTestPackage;
use App\Models\SaleInvoiceSubDetails;
use App\Models\SpecialCase;
use App\Models\FinancialYear;
use App\Models\AcJournal;
use App\Models\AcJournalTransaction;
use App\Models\MenuModelPermission;
use App\Models\VoucherMaster;
use App\Models\AccountTransaction;
use App\Models\SaleInvoiceBatch;
use App\Models\Company;
use App\Models\User;
use App\Models\VoucherType;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class AccountingJournalsController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:journal-voucher-list', ['only' => ['index', 'store']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!$this->modulePermissionCheck('JOURNAL_VOUCHER')) {
            abort(404);
        }

        $data = AcJournal::select([
                'ac_journals.created_at',
                'ac_journals.voucher_no',
                'ac_journals.transaction_date',
                'ac_journals.receipt_time',
                'ac_journals.reference',
                'ac_journals.description',
                'ac_journals.voucher_approval_status',
                'ac_journals.voucher_status',
                'ac_journals.id',
                'companies.name',
                'tbl_financial_year.financial_year_name',
                'ac_journals.voucher_id',

            ])
            ->leftjoin('tbl_financial_year','tbl_financial_year.financialyearid','ac_journals.financial_year_id')
            ->leftjoin('companies','companies.id','ac_journals.company_id');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');
            $search_type = $request->get('search_type');

            $data = $data->when(!empty($search_query) && !empty($search_type),
                function ($query) use ($search_query, $search_type) {
                    if ($search_type == 'voucher_no') {
                        $query->where('ac_journals.voucher_no', 'like', '%'.$search_query.'%');
                    } else if ($search_type == 'company') {
                        $query->where('companies.name', 'like', '%'.$search_query.'%');
                    } else if ($search_type == 'description') {
                        $query->where('ac_journals.description', 'like', '%'.$search_query.'%');
                    } else if ($search_type == 'financial_year') {
                        $query->where('tbl_financial_year.financial_year_name', 'like', '%'.$search_query.'%');
                    }
                }
            )
            ->latest()
            ->paginate($sort_by);

            return view('accounting-journals.table', compact('data'));
        }
        else
        {
            $data = $data->latest()
                        ->paginate(10);

            return view('accounting-journals.index',compact('data'));
        }
        //
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $company_address = \Session::get('company_data')['companies_addresses'];
        $countryCode     = $company_address->country_code ?? 'us';

        $brands          = ProductBrand::select(['id','name','is_default'])->where(['status' => 1])->get();
        $financialYears = \Session::get('financialyear');
        $companyDateFormate = phpToJsDateFormat($this->companyDateFormate());

        return view('accounting-journals.create', compact('financialYears','countryCode','brands','companyDateFormate'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function journalVoucherUpdate(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];

        if (empty($data) || ((!isset($data['edit_journal_voucher_entries']) ||
            count($data['edit_journal_voucher_entries']) < 1) && (!isset($data['add_journal_voucher_entries']) ||
            count($data['add_journal_voucher_entries']) < 1))
        ) {
            $data = [];
            \Session::put('journal_voucher_details', $data);
            return response()->json([ 'success' => false, 'message'=> 'No Journal Voucher Entry Found!' ]);
        }

        $item = [
            'reference'         => $request->voucher_no,
            'transaction_date'  => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
            'description'       => $request->description,
            'financial_year_id' => $request->financial_year_id
        ];

        $journalVoucher           = AcJournal::where('id', '=', $request->id)->update($item);
        $oldJournalTransaction    = AccountTransaction::where([
            'voucher_id' => $request->id,
            'module_code' => 'JOURNAL_VOUCHER'
        ])->first();

        $oldJournalTransactionIds = [];

         if (!empty($data) || isset($data['edit_journal_voucher_entries']) || count($data['edit_journal_voucher_entries']) > 0) {
            foreach ($data['edit_journal_voucher_entries'] as $item) {
                $item1 = [
                   'transactions_no'     => $item['transactions_no'],
                   'transaction_date'    => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
                   'module_id'           => $item['module_id'],
                   'voucher_type'        => $item['voucher_type'],
                   'account_id'          => $item['account_id'],
                   'transaction_type'    => $item['transaction_type'],
                   'details_narration'   => $item['details_narration'],
                   'amount'              => $item['amount'],
                   'currency_id'         => $item['currency_id'],
                   'created_by'          => $item['created_by'],
                   'company_id'          => $item['company_id'],
                   'financial_year_id'   => $item['financial_year_id'],
                ];

                $acJournalTransaction       = AcJournalTransaction::where('id', $item['id'])->update($item1);
                $oldJournalTransactionIds[] = $item['id'];

                AccountTransaction::updateTransaction([
                    'module_code'      => 'JOURNAL_VOUCHER',
                    'voucher_type'     => $item['voucher_type'], // SALE, PURCHASE, PAYMENT
                    'account_id'       => $item['account_id'],
                    'transaction_type' => $item['transaction_type'],
                    'transactions_no'  => $oldJournalTransaction->transactions_no,
                    'voucher_id'       => $request->id,
                    'voucher_sub_id'   => $item['id'],
                ],[
                    'amount'            => $item['amount'],
                    'financial_year_id' => $request->financial_year_id,
                ]);
            }
        }

        AcJournalTransaction::whereNotIn('id',$oldJournalTransactionIds)->delete();

        if (isset($data['add_journal_voucher_entries'])) {
            foreach ($data['add_journal_voucher_entries'] as $item) {
                $item1 = [
                   'transactions_no'     => $item['transactions_no'],
                   'transaction_date'    => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
                   'module_id'           => $item['module_id'],
                   'voucher_id'          => $request->id,
                   'voucher_type'        => $item['voucher_type'],
                   'account_id'          => $item['account_id'],
                   'transaction_type'    => $item['transaction_type'],
                   'details_narration'   => $item['details_narration'],
                   'amount'              => $item['amount'],
                   'currency_id'         => $item['currency_id'],
                   'created_by'          => $item['created_by'],
                   'company_id'          => $item['company_id'],
                   'financial_year_id'   => $item['financial_year_id'],
                   'transactions_status' => true,
                ];

                $acJournalTransaction = AcJournalTransaction::create($item1);

                AccountTransaction::createTransaction([
                    'transactions_no'   => $oldJournalTransaction->transactions_no,
                    'transaction_date'  => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
                    'module_code'       => 'JOURNAL_VOUCHER',
                    'voucher_id'        => $request->id,
                    'voucher_sub_id'    => $acJournalTransaction->id,
                    'voucher_type'      => $item['voucher_type'],
                    'account_id'        => $item['account_id'],
                    'transaction_type'  => $item['transaction_type'],
                    'details_narration' => $item['details_narration'],
                    'amount'            => $item['amount'],
                    'financial_year_id' => $request->financial_year_id,
                ]);
            }
        }

        \Session::put('journal_voucher_details', []);
         // }
         return response()->json([ 'success' => true,'message'=> 'Journal Voucher Updated Successfully!' ]);
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function journalVoucherSubmit(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];
        if (empty($data) || !isset($data['journal_voucher_entries']) || count($data['journal_voucher_entries']) < 1) {
            $data = [];
            \Session::put('journal_voucher_details', $data);
            return response()->json([ 'success' => false, 'message'=> 'No Journal Voucher Entry Found!' ]);
            //return redirect()->route('accounting-journals.create')->with('error', 'No journal voucher entry found!');
        }

        $voucher_master = VoucherMaster::with('voucherSeries')->where('voucher_code', 'JOURNAL')->where(['status' => 1])->first();

        # last voucher count for main invoice number
        $voucher_count      = AcJournal::select('last_voucher_count')->orderBy('created_at', 'DESC')->first();
        $last_voucher_count = isset($voucher_count) && isset($voucher_count->last_voucher_count) ? $voucher_count->last_voucher_count + 1 : ($voucher_master->voucherSeries->start_from ?? 0) + 1;

        $main_invoice_no = $this->generateCode(
            $last_voucher_count,
            ($voucher_master->voucherSeries->prefix ?? 'RV'),
            ($voucher_master->voucherSeries->postfix ?? ''),
            ($voucher_master->voucherSeries->separator ?? '-'),
            ($voucher_master->voucherSeries->length ?? 5)
        );

        $item = [
            'voucher_no'         => $main_invoice_no,
            'reference'          => $request->voucher_no,
            'transaction_date'   => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
            'description'        => $request->description ?? '',
            'company_id'         => \Auth::user()->company_id,
            'financial_year_id'  => $request->financial_year_id,
            'voucher_status'     => true,
            'created_by'         => \Auth::user()->id ?? '',
            'last_voucher_count' => $last_voucher_count,
            'voucher_master_id'  => ($voucher_master->id ?? ''),
            'receipt_time'        => Carbon::now(),
        ];

        $journalVoucher = AcJournal::create($item);
        $transactions_no = $this->generateRandomCode();

        $transactionsData = $this->generateTransactionNumber();

        if (!empty($data) || isset($data['journal_voucher_entries']) || count($data['journal_voucher_entries']) > 0) {
            foreach ($data['journal_voucher_entries'] as $item) {
                $item1 = [
                   'transactions_no'     =>$transactionsData['transactions_no'],
                   'transaction_date'    =>$request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
                   'module_id'           =>$item['module_id'],
                   'voucher_id'          =>$journalVoucher->id,
                   'voucher_type'        =>$item['voucher_type'],
                   'account_id'          =>$item['account_id'],
                   'transaction_type'    =>$item['transaction_type'],
                   'details_narration'   =>$item['details_narration'],
                   'amount'              =>$item['amount'],
                   'currency_id'         =>$item['currency_id'],
                   'created_by'          =>$item['created_by'],
                   'company_id'          =>$item['company_id'],
                   'financial_year_id'   =>$item['financial_year_id'],
                   'transactions_status' => true,
                    'financial_year_id' => $request->financial_year_id,
                ];
                $acJournalTransaction = AcJournalTransaction::create($item1);

                AccountTransaction::createTransaction([
                    'transactions_no'   => $transactionsData['transactions_no'],
                    'transaction_date'  => $request->transaction_date?Carbon::createFromFormat($this->companyDateFormate(), $request->transaction_date)->format('Y-m-d'):date('Y-m-d'),
                    'module_code'       => 'JOURNAL_VOUCHER',
                    'voucher_id'        => $journalVoucher->id,
                    'voucher_sub_id'    => $acJournalTransaction->id,
                    'voucher_type'      => $item['voucher_type'],
                    'account_id'        => $item['account_id'],
                    'transaction_type'  => $item['transaction_type'],
                    'details_narration' => $item['details_narration'],
                    'amount'            => $item['amount'],
                    'financial_year_id' => $request->financial_year_id,
                    'last_id'           => $transactionsData['last_count'],
                ]);
            }
            \Session::put('journal_voucher_details', []);
         }
         return response()->json([ 'success' => true,'message'=> 'New Journal Voucher Created Successfully!' ]);
        //return redirect()->route('accounting-journals.index')->with('success', 'New Journal Voucher Created Successfully!');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $data          = \Session::get('billing_products_cart') ?? [];
        $company_id    = \Auth::user()->company_id ?? '';
        $payment_terms = ($data['due_balance_amount'] < 1) ? 'CASH' : 'CREDIT';
        $sale_invoice  = [
            'type_id'                => $request->invoice_type_id,
            'payment_mode_id'        => $request->payment_mode,
            'totalbillingamount'     => $data['amount'],
            'date'                   => $data['sample_receive_date'],
            'patient_id'             => $data['patient_id'],
            'doctor_id'              => $data['doctor_id'],
            'collection_agent_id'    => $data['collection_id'],
            'refferal_lab_id'        => $data['refferal_id'],
            'reff_no'                => $request->reff_no,
            'amountreceived'         => $request->amount_recieved,
            'duebalanceamount'       => $data['due_balance_amount'],
            'sale_delivery_date'     => date('Y-m-d'),
            'sale_delivery_status'   => 'PENDING',
            'company_id'             => $company_id,
            'status'                 => 'NEWSALE',
            'remarks'                => $request->sale_remarks,
            'special_case_id'        => $request->special_case,
            'amountadjustment'       => $request->special_discount,
            'adjustment_type'        => $request->discount_type,
            'financial_year'         => $this->financialyear(),
            'payment_terms'          => $payment_terms,
            'industry_type'          => 'PATHOLOGY',
            'operator_id'            => \Auth::ID(),
        ];

        $sale_invoice       = SaleInvoice::create($sale_invoice);
        $sale_invoice_id    = $this->generateCode($sale_invoice->id, 'INV');
        $sale_invoice->update(['enquiry_id' => $sale_invoice_id]);

        //sale invoice batch generate number
        $voucher_master = VoucherMaster::with('voucherSeries')->where('voucher_code', 'Sales_Invoice_Batch')->first();
        $voucher_count_batch      = SaleInvoiceBatch::select('last_voucher_count')->orderBy('created_at', 'DESC')->first();
        // dd($voucher_count_batch);

        $last_voucher_count_batch = isset($voucher_count_batch) && isset($voucher_count_batch->last_voucher_count) ? $voucher_count_batch->last_voucher_count + 1 : ($voucher_master->voucherSeries->start_from + 1 ?? 1);

        $main_invoice_no = $this->generateCode(
            $last_voucher_count_batch,
            ($voucher_master->voucherSeries->prefix ?? 'BAT'),
            ($voucher_master->voucherSeries->postfix ?? ''),
            ($voucher_master->voucherSeries->separator ?? '-'),
            ($voucher_master->voucherSeries->length ?? 5)
        );

        $invoice_batch = [
            'account_id'             => '',
            'invoice_batch_date'     => Carbon::now(),
            'invoice_batch_no'       => $main_invoice_no,
            'created_by'             => Auth::user()->id,
            'invoice_id'             => $sale_invoice->id,
            'last_voucher_count'     => $last_voucher_count_batch,
            'sample_barcode'         => $request->sample_barcode,
            'sample_receive_date'    => $request->sample_receive_date,
            'sample_receive_time'    => $request->sample_receive_time,
            'remarks_for_technician' => $request->remarks_for_technician,
        ];
        //sale invoice batch
        $invoice_batch  = SaleInvoiceBatch::create($invoice_batch);

        foreach($data['products'] as $key => $value) {
            $product = Product::select(['attribute_id','warehouse_id'])
                        ->where('id', $value['product_id'])
                        ->first();
            $product_price = ProductPrice::select(['batch'])
                        ->where('id', $value['price_id'])
                        ->first();
            $item = [
                'sale_invoice_id'      => $sale_invoice->id,
                'product_id'           => $value['product_id'],
                'item_code'            => $product->shortcode,
                'product_price_id'     => $value['price_id'],
                'sale_invoice_date'    => date('Y-m-d'),
                'item_price'           => $value['product_price'],
                'item_basic_amount'    => $value['product_price'],
                'item_discount'        => $value['per_discount'],
                'item_discount_amount' => $value['flat_discount'],
                'item_total_amount'    => $value['amount'],
                'item_details1'        => $value['product_details'],
                'item_batch'           => $product_price->batch ?? '',
                'item_details1'        => $product->description ?? '',
                'item_mqty'            => $product_price->item_min_stock_qty ?? '',
                'item_tax'             => $product->tax_slab_id ?? '',
                'delivery_status'      => 'PENDING',
            ];

            $sale_invoice_details = SaleInvoiceDetail::create($item);
            $product_test_package = ProductTestPackage::where('product_id', $value['product_id'])->get();

            foreach ($product_test_package as $package) {
                $addedTestPackage = SaleInvoiceSubDetails::create([
                    'invoice_details_id'         => $sale_invoice_details->id,
                    'invoice_id'                 => $sale_invoice->id,
                    'testpackage_id'             => $package->test_package_id,
                    'invoice_subdetails_remarks' => '',
                    'invoice_subdetails_status'  => 'SAMPLE_COLLECTION',
                ]);
            }
        }
        return redirect()->route('invoice-print', $sale_invoice_id)->with('success', "New Bill Created Successfully");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $accountJournal = AcJournal::with('financialYear','company','accountJournalTransaction','accountJournalTransaction.financialYear','accountJournalTransaction.company','accountJournalTransaction.user','accountJournalTransaction.account','accountJournalTransaction.module')->where(['id'=>$id])->first();
        return view('accounting-journals.show', compact('accountJournal'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {

        $data101 = \Session::get('journal_voucher_details') ?? [];
        /******************************************************************/
        /******************************************************************/
        $data = AcJournal::find($id);
        $acJournalTransaction = AcJournalTransaction::select(
                                'ac_journal_transactions.*',
                                'accounts.name as account_name',
                                \DB::raw("IF(ac_journal_transactions.transaction_type = 'DEBIT', ac_journal_transactions.amount ,0) as debit"),
                                \DB::raw("IF(ac_journal_transactions.transaction_type = 'CREDIT', ac_journal_transactions.amount ,0) as credit")
                            )
                            ->leftjoin('accounts', 'accounts.id', 'ac_journal_transactions.account_id')
                            ->where(['ac_journal_transactions.voucher_id'=>$id])
                            ->get();
        $data101['edit_journal_voucher_entries']= isset($acJournalTransaction)? $acJournalTransaction->toArray() : [];
        \Session::put('journal_voucher_details', $data101);
        /******************************************************************/
        /******************************************************************/

        $company_address = \Session::get('company_data')['companies_addresses'];
        $countryCode     = $company_address->country_code ?? 'us';

        $brands         = ProductBrand::select(['id','name','is_default'])->where(['status' => 1])->get();
        /*$financialYears = FinancialYear::select(['financialyearid','financial_year_name','is_default'])
            ->where(['status' => '1'])
            ->orderBy('financialyearid','desc')
            ->get();*/
        $financialYears = \Session::get('financialyear');

        return view('accounting-journals.edit', compact('data','financialYears','countryCode','brands'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {

        $data = AcJournal::find($request->id);
        $data->delete();

        AcJournalTransaction::where(['voucher_id'=>$request->id])->delete();

        AccountTransaction::where([
            'voucher_id'  => $request->id,
            'module_code' => 'JOURNAL_VOUCHER'
        ])->delete();

        $Redirect = 'journal-voucher';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }

    public function searchAccounts(Request $request)
    {
        $searchTerm = $request->search_text;
        $results    = Account::select(['accounts.id','accounts.name'])
                    ->where('accounts.name', 'LIKE', '%' . $searchTerm . '%')
                    ->get();
        return response()->json(['result' => $results, 'status' => true]);
    }

    public function journalVoucherEntries(Request $request)
    {
        $page = isset($request->page) && $request->page == 'billing' ? 1 : 0;
        $data = \Session::get('journal_voucher_details') ?? [];
        return view('accounting-journals.journal-voucher-entries', compact('data', 'page'));
    }

    public function editJournalVoucherEntries(Request $request)
    {
        $page = isset($request->page) && $request->page == 'billing' ? 1 : 0;
        $data = \Session::get('journal_voucher_details') ?? [];

        return view('accounting-journals.edit-journal-voucher-entries', compact('data', 'page'));
    }

    public function addJournalVouchers(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];

        if (!isset($data['voucher_no']) || $data['voucher_no'] == '') {
            $data['voucher_no'] = $request->voucher_no;
        }
        if (!isset($data['transaction_date']) || $data['transaction_date'] == '') {
            $data['transaction_date'] = $request->transaction_date;
        }

        if (!isset($data['financial_year_id']) || $data['financial_year_id'] == '') {
            $data['financial_year_id'] = $request->financial_year_id ?? '';
        }

        if (!isset($data['account_id']) || $data['account_id'] == '') {
            $data['account_id'] = $request->account_id ?? '';
        }

        if (!isset($data['account_name']) || $data['account_name'] == '') {
            $data['account_name'] = $request->account_name ?? '';
        }

        if (!isset($data['debit']) || $data['debit'] == '') {
            $data['debit'] = $request->debit ?? 0;
        }

        if (!isset($data['credit']) || $data['credit'] == '') {
            $data['credit'] = $request->credit ?? '0';
        }

        if (!isset($data['details_narration']) || $data['details_narration'] == '') {
            $data['details_narration'] = $request->details_narration ?? '';
        }

        if (!isset($data['description']) || $data['description'] == '') {
            $data['description'] = $request->description ?? '';
        }

        $company_address = \Session::get('company_data')['companies_addresses'];
        $countryCurrencyId = $company_address->currency_id ?? '0';

        $module            = MenuModelPermission::where('code','JOURNAL_VOUCHER')->first();
        $module_id         = $module->id ?? '';

        $data['journal_voucher_entries'][] = [
            'transactions_no'       => $this->generateRandomCode(),
            'transaction_date'      => isset($request->transaction_date) ? date('Y-m-d', strtotime($request->transaction_date)) : '',
            'module_id'             => $module_id,
            'voucher_type'          => 'JOURNAL',
            'account_id'            => $request->account_id ?? '',
            'transaction_type'      => (isset($request->debit) && $request->debit>0 ? 'DEBIT': 'CREDIT'),
            'details_narration'     => $request->details_narration ?? '',
            'amount'                => (isset($request->debit) && $request->debit>0 ? $request->debit : $request->credit),
            'created_by'            => \Auth::user()->id ?? '',
            'company_id'            => \Auth::user()->company_id ?? '',
            'currency_id'           => $countryCurrencyId,
            'financial_year_id'     => $request->financial_year_id ?? '',
            'debit'                 => $request->debit ?? '',
            'credit'                => $request->credit ?? '',
            'transactions_status'   => true,
            'account_name'          => $request->account_name ?? '',
        ];
        \Session::put('journal_voucher_details', $data);

        return response()->json([ 'success' => true,'message'=> 'New Journal Voucher Entry Added Successfully!'  ]);
    }

    public function editJournalVouchers(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];

        if (!isset($data['voucher_no']) || $data['voucher_no'] == '') {
            $data['voucher_no'] = $request->voucher_no;
        }
        if (!isset($data['transaction_date']) || $data['transaction_date'] == '') {
            $data['transaction_date'] = isset($request->transaction_date) ? date('Y-m-d', strtotime($request->transaction_date)) : '';
        }

        if (!isset($data['financial_year_id']) || $data['financial_year_id'] == '') {
            $data['financial_year_id'] = $request->financial_year_id ?? '';
        }

        if (!isset($data['account_id']) || $data['account_id'] == '') {
            $data['account_id'] = $request->account_id ?? '';
        }

        if (!isset($data['account_name']) || $data['account_name'] == '') {
            $data['account_name'] = $request->account_name ?? '';
        }

        if (!isset($data['debit']) || $data['debit'] == '') {
            $data['debit'] = $request->debit ?? 0;
        }

        if (!isset($data['credit']) || $data['credit'] == '') {
            $data['credit'] = $request->credit ?? '0';
        }

        if (!isset($data['details_narration']) || $data['details_narration'] == '') {
            $data['details_narration'] = $request->details_narration ?? '';
        }

        if (!isset($data['description']) || $data['description'] == '') {
            $data['description'] = $request->description ?? '';
        }

        $company_id      = \Auth::user()->company_id ?? '';
        
        $company_address = \Session::get('company_data')['companies_addresses'];
        $countryCurrencyId = $company_address->currency_id ?? '0';

        $module            = MenuModelPermission::where('code','JOURNAL_VOUCHER')->first();
        $module_id         = $module->id ?? '';

        $data['add_journal_voucher_entries'][] = [
            'transactions_no'       => $this->generateRandomCode(),
            'transaction_date'      => isset($request->transaction_date) ? date('Y-m-d', strtotime($request->transaction_date)) : '',
            'module_id'             => $module_id,
            'voucher_type'          => 'JOURNAL',
            'account_id'            => $request->account_id ?? '',
            'transaction_type'      => (isset($request->debit) && $request->debit>0 ? 'DEBIT': 'CREDIT'),
            'details_narration'     => $request->details_narration ?? '',
            'amount'                => (isset($request->debit) && $request->debit>0 ? $request->debit : $request->credit),
            'created_by'            => \Auth::user()->id ?? '',
            'company_id'            => \Auth::user()->company_id ?? '',
            'currency_id'           => $countryCurrencyId,
            'financial_year_id'     => $request->financial_year_id ?? '',
            'debit'                 => $request->debit ?? '',
            'credit'                => $request->credit ?? '',
            'transactions_status'   => true,
            'account_name'          => $request->account_name ?? '',
        ];
        \Session::put('journal_voucher_details', $data);

        return response()->json([ 'success' => true,'message'=> 'New Journal Voucher Entry Added Successfully!'  ]);
    }

    public function cartProductRemove(Request $request)
    {
        $data = \Session::get('billing_products_cart') ?? [];
        $id   = $request->id;

        $amount = $data['amount'];

        $data['products'] = array_filter($data['products'], function ($product) use ($id, &$data, &$amount) {
            if ($product['product_id'] == $id) {
                $amount = $data['amount'] - $product['amount'];
            }

            return $product['product_id'] != $id;
        });

        if (count($data['products']) == 0) {
            $data['amount'] = 0;
        } else {
            $data['amount'] = $amount;
        }

        \Session::put('billing_products_cart', $data);

        return response()->json([ 'success' => true ]);
    }

    public function journalVoucherEntryUpdate(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];

        foreach ($data['journal_voucher_entries'] as $key => $value) {
            if ($key == $request->id && isset($request->details_narration)) {
                $data['journal_voucher_entries'][$key]['details_narration'] = $request->details_narration;
            }
            if ($key == $request->id && isset($request->amount)) {
                $data['journal_voucher_entries'][$key]['amount'] = $request->amount;
            }
            if ($key == $request->id && isset($request->debit)) {
                $data['journal_voucher_entries'][$key]['debit'] = $request->debit;
                $data['journal_voucher_entries'][$key]['amount'] = $request->debit;
            }
            if ($key == $request->id && isset($request->credit)) {
                $data['journal_voucher_entries'][$key]['credit'] = $request->credit;
                $data['journal_voucher_entries'][$key]['amount'] = $request->credit;
            }
        }

        \Session::put('journal_voucher_details', $data);

        return response()->json([ 'success' => true , 'data' => $data]);
    }

    public function editJournalVoucherEntryUpdate(Request $request)
    {
        $data = \Session::get('journal_voucher_details') ?? [];

        if ($request->edit == 1) {
            foreach ($data['edit_journal_voucher_entries'] as $key => $value) {
                if ($key == $request->id && isset($request->details_narration)) {
                    $data['edit_journal_voucher_entries'][$key]['details_narration'] = $request->details_narration;
                }
                if ($key == $request->id && isset($request->amount)) {
                    $data['edit_journal_voucher_entries'][$key]['amount'] = $request->amount;
                }
                if ($key == $request->id && isset($request->debit)) {
                    $data['edit_journal_voucher_entries'][$key]['debit'] = $request->debit;
                    $data['edit_journal_voucher_entries'][$key]['amount'] = $request->debit;
                }
                if ($key == $request->id && isset($request->credit)) {
                    $data['edit_journal_voucher_entries'][$key]['credit'] = $request->credit;
                    $data['edit_journal_voucher_entries'][$key]['amount'] = $request->credit;
                }
            }
        } else {
            foreach ($data['add_journal_voucher_entries'] as $key => $value) {
                if ($key == $request->id && isset($request->details_narration)) {
                    $data['add_journal_voucher_entries'][$key]['details_narration'] = $request->details_narration;
                }
                if ($key == $request->id && isset($request->amount)) {
                    $data['add_journal_voucher_entries'][$key]['amount'] = $request->amount;
                }
                if ($key == $request->id && isset($request->debit)) {
                    $data['add_journal_voucher_entries'][$key]['debit'] = $request->debit;
                    $data['add_journal_voucher_entries'][$key]['amount'] = $request->debit;
                }
                if ($key == $request->id && isset($request->credit)) {
                    $data['add_journal_voucher_entries'][$key]['credit'] = $request->credit;
                    $data['add_journal_voucher_entries'][$key]['amount'] = $request->credit;
                }
            }
        }

        \Session::put('journal_voucher_details', $data);

        return response()->json([ 'success' => true , 'data' => $data]);
    }

    public function print($id)
    {
        return view('lab-billings.print');
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('voucher_status' => $request->status );
            $Update = AcJournal::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Journal Voucher status successfully change'],
                    'data'=>[
                       'redirect'=>'/journal-voucher/',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change status'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }

    public function changeApprovalStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('voucher_approval_status' => $request->status );
            $Update = AcJournal::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Journal Voucher approval status successfully change'],
                    'data'=>[
                       'redirect'=>'/journal-voucher/',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change approval status'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }


    public function journalVoucherEntryRemove(Request $request)
    {
        $index   = $request->id;
        $event_data_display = \Session::get('journal_voucher_details');

        unset($event_data_display['journal_voucher_entries'][$index]);
        \Session::put('journal_voucher_details', $event_data_display);

        return response()->json([ 'success' => true,'message'=> 'New Journal Voucher Entry Removed Successfully!']);
    }


    public function editJournalVoucherEntryRemove(Request $request)
    {
        $index              = $request->id;
        $type               = $request->type;
        $event_data_display = \Session::get('journal_voucher_details');

        if ($type == 'edit') {
            unset($event_data_display['edit_journal_voucher_entries'][$index]);
        } else {
            unset($event_data_display['add_journal_voucher_entries'][$index]);
        }

        \Session::put('journal_voucher_details', $event_data_display);

        return response()->json([ 'success' => true,'message'=> 'Edit Journal Voucher Entry Removed Successfully!','data'=>$event_data_display]);
    }

    public function printJournalVoucher($id)
    {
        $accountJournal = AcJournal::with(
            'company',
            'financialYear',
            'voucherMaster',
            'accountJournalTransaction',
            'accountJournalTransaction.financialYear',
            'accountJournalTransaction.company',
            'accountJournalTransaction.account',
            'accountJournalTransaction.module',
            'accountJournalTransaction.user',
        )
            ->where([ 'id' => $id ])
            ->first();

        return view('invoice-formats.journal-voucher.journal-voucher-print', compact('accountJournal'));
        $pdfData  = [
            'accountJournal' => $accountJournal,
        ];

        $pdf = PDF::loadView('invoice-formats.journal-voucher.journal-voucher-print', $pdfData)->set_option("isPhpEnabled", true)
            ->setPaper('a4', 'portrait')
            ->setWarnings(false);

        $filename = "journal-voucher_" . date("Y_m_d_i_H") . ".pdf";

        return $pdf->download($filename);
    }

    public function journalVoucherStatement(Request $request)
    {

        $company = Company::find(Auth::user()->company_id);
        $companyDateFormate = phpToJsDateFormat($this->companyDateFormate());
        if ($request->ajax()) {
        $from_date       = $request->from_date?Carbon::createFromFormat($this->companyDateFormate(), $request->from_date)->format('Y-m-d'):date('Y-m-d');
        $to_date         = $request->to_date?Carbon::createFromFormat($this->companyDateFormate(), $request->to_date)->format('Y-m-d'):date('Y-m-d');
        $branch    = $request->branch ?? '';
        $account   = $request->account ?? '';
        $created_by      = $request->created_by??'';

        $accountJournal = $this->queryFilter($request);
        return view('invoice-formats.journal-voucher.statement-table', compact('company','from_date','to_date','accountJournal'));
        }
        
       $voucher_type = VoucherType::where('code', 'JOURNAL')->where(['status' => 1])->first();

        return view('invoice-formats.journal-voucher.statement', compact(
            'company',
            'voucher_type',
            'companyDateFormate'
        ));
    }

    public function exportStatement(Request $request)
    {
        $from_date       = $request->from_date?Carbon::createFromFormat($this->companyDateFormate(), $request->from_date)->format('Y-m-d'):date('Y-m-d');
        $to_date         = $request->to_date?Carbon::createFromFormat($this->companyDateFormate(), $request->to_date)->format('Y-m-d'):date('Y-m-d');
        $data = $this->queryFilter($request);
        $fileName = 'Journal-Voucher-Statements'.' From '.$from_date.' To '.$to_date.'.xlsx';
        return Excel::download(new JournalVoucherExport($data->toArray()), $fileName);
    }

    public  function queryFilter(Request $request){
        $from_date          = $request->from_date?Carbon::createFromFormat($this->companyDateFormate(), $request->from_date)->format('Y-m-d'):date('Y-m-d');
        $to_date            = $request->to_date?Carbon::createFromFormat($this->companyDateFormate(), $request->to_date)->format('Y-m-d'):date('Y-m-d');
        $branch             = $request->branch ?? '';
        $account            = $request->account ?? '';
        $created_by         = $request->created_by??'';
        $paymentMode        = $request->payment_mode ?? '';
        $voucher_type_search= $request->voucher_type??'';
        $accountJournal = AcJournal::with(
                /*'company',
                'financialYear',
                'voucherMaster',*/
                'accountJournalTransaction',
                'accountJournalTransaction.financialYear',
                'accountJournalTransaction.company',
                'accountJournalTransaction.account',
                'accountJournalTransaction.module',
                'accountJournalTransaction.user',
            )
             ->when($from_date != '' && $to_date != '', function($query) use($from_date, $to_date) {
                $query->whereBetween('ac_journals.transaction_date', [
                    date('Y-m-d', strtotime($from_date)),
                    date('Y-m-d', strtotime($to_date))
                ]);
            })
            ->when($branch != '', function($query) use($branch) {
                $query->where('ac_journals.company_id', $branch);
            })
            ->when($created_by != '', function($query) use($created_by) {
                $query->where('ac_journals.created_by', $created_by);
            })
            ->when($account != '', function($query) use($account) {
                $query->whereHas('accountJournalTransaction', function ($query) use ($account) {
                    $query->where('ac_journal_transactions.account_id',$account);
                });
            })
            ->get();
        return $accountJournal;
    }
}
