<?php

namespace App\Http\Controllers\Accounts;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\BloodGroup;

class BloodGroupsController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:blood-groups-list', ['only' => ['index']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!$this->modulePermissionCheck('BLOOD_GROUPS')) {
            abort(404);
        }

        $data = BloodGroup::paginate(10);

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = BloodGroup::when(!empty($search_query), function ($query) use ($search_query) {
                            return $query->where('group', 'like', '%'.$search_query.'%');
                        })
                        ->paginate($sort_by);

            return view('blood-groups.table', compact('data'));
        }
        else
        {
            return view('blood-groups.index',compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('blood-groups.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'group' => 'required|unique:blood_groups'
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $blood = BloodGroup::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Blood Group created successfully',
                'data'    => $blood
            ]);
        }

        return redirect()->route('blood-groups.index')
            ->with('success','Blood Group created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $group = BloodGroup::find($id);

        return view('blood-groups.edit',compact('group'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'group' => 'required|unique:blood_groups,group,'.$id
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $group = BloodGroup::find($id);

        if (!$group) {
            return redirect()->route('blood-groups.index')
                ->with('error','Blood Group not found!');
        }

        $group->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Blood Group updated successfully',
                'data'    => $group
            ]);
        }

        return redirect()->route('blood-groups.index')
            ->with('success','Blood Group updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        BloodGroup::find($request->id)->delete();

        $Redirect = 'blood-groups';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
