<?php

namespace App\Http\Controllers\Accounts;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\MaritalStatus;

class MaritalStatusesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:marital-status-list', ['only' => ['index']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!$this->modulePermissionCheck('MARITAL_STATUS')) {
            abort(404);
        }

        $data = MaritalStatus::paginate(10);

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = MaritalStatus::when(!empty($search_query), function ($query) use ($search_query) {
                            return $query->where('name', 'like', '%'.$search_query.'%');
                        })
                        ->paginate($sort_by);

            return view('marital-statuses.table', compact('data'));
        }
        else
        {
            return view('marital-statuses.index',compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('marital-statuses.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'name' => 'required|unique:marital_statuses'
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        MaritalStatus::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Marital Status created successfully',
            ]);
        }

        return redirect()->route('marital-statuses.index')
            ->with('success','Marital Status created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $marital_statuses = MaritalStatus::find($id);

        return view('marital-statuses.edit',compact('marital_statuses'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'name' => 'required|unique:marital_statuses,name,'.$id
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $marital_statuses = MaritalStatus::find($id);

        if (!$marital_statuses) {
            return redirect()->route('marital-statuses.index')
                ->with('error','Marital Status not found!');
        }

        $marital_statuses->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Marital Status updated successfully',
            ]);
        }

        return redirect()->route('marital-statuses.index')
            ->with('success','Marital Status updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        MaritalStatus::find($request->id)->delete();

        $Redirect = 'marital-statuses';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
