<?php

namespace App\Http\Controllers\Company;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CompanyEstablishmentType;

class CompanyEstablishmentTypesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:company-establishment-types-list', ['only' => ['index']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!$this->modulePermissionCheck('COMPANY_ESTABLISHMENT_TYPES')) {
            abort(404);
        }

        $data = CompanyEstablishmentType::paginate(10);

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = CompanyEstablishmentType::when(!empty($search_query), function ($query) use ($search_query) {
                    return $query->where('name', 'like', '%'.$search_query.'%');
                })
                ->paginate($sort_by);

            return view('establishment-types.table', compact('data'));
        }
        else
        {
            return view('establishment-types.index',compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('establishment-types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'name' => 'required|unique:company_establishment_types'
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $establishment_types = CompanyEstablishmentType::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Company Establishment Type created successfully',
                'data'    => $establishment_types
            ]);
        }

        return redirect()->route('establishment-types.index')
            ->with('success','Company Establishment Type created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $establishment_types = CompanyEstablishmentType::find($id);

        return view('establishment-types.edit',compact('establishment_types'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'name' => 'required|unique:company_establishment_types,name,'.$id
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $establishment_types = CompanyEstablishmentType::find($id);

        if (!$establishment_types) {
            return redirect()->route('establishment-types.index')
                ->with('error','Company Establishment Type not found!');
        }

        $establishment_types->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Company Establishment Type updated successfully',
                'data'    => $establishment_types
            ]);
        }

        return redirect()->route('establishment-types.index')
            ->with('success','Company Establishment Type updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        CompanyEstablishmentType::find($request->id)->delete();

        $Redirect = 'establishment-types';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
