<?php

namespace App\Http\Controllers\Enquiries;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\EnquiryQuality;

class EnquiryQualitiesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:enquiry-qualities-list', ['only' => ['index']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $data = EnquiryQuality::select([
                    'id',
                    'code',
                    'enquiry_quality_name',
                    'enquiry_quality_description',
                    'enquiry_quality_priority',
                    'color',
                    'bg_color',
                    'default',
                    'status'
                ]);

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                        return $query->where('enquiry_quality_name', 'like', '%'.$search_query.'%')
                            ->orWhere('enquiry_quality_description', 'like', '%'.$search_query.'%');
                    })
                    ->paginate($sort_by);

            return view('enquiry-qualities.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('enquiry-qualities.index',compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $priority = EnquiryQuality::max('enquiry_quality_priority');
        $priority = $priority != '' ? $priority : 0;

        return view('enquiry-qualities.create',compact('priority'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'enquiry_quality_name'     => 'required|unique:enquiry_qualities',
            'enquiry_quality_priority' => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $data['company_id'] = \Auth::user()->company_id;

        $default_check = EnquiryQuality::where(['default' => 1,'status' => 1])->first();

        if (!$default_check) {
            $data['default'] = 1;
        }

        $quality = EnquiryQuality::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Enquiry Quality created successfully',
                'data'    => $quality
            ]);
        }

        return redirect()->route('enquiry-qualities.index')
            ->with('success','Enquiry Quality created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $quality = EnquiryQuality::find($id);

        return view('enquiry-qualities.edit',compact('quality'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'enquiry_quality_name'     => 'required|unique:enquiry_qualities,enquiry_quality_name,'.$id,
            'enquiry_quality_priority' => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $quality = EnquiryQuality::find($id);

        if (!$quality) {
            return redirect()->route('enquiry-qualities.index')
                ->with('error','Enquiry Quality not found!');
        }

        $quality->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Enquiry Quality updated successfully',
                'data'    => $quality
            ]);
        }

        return redirect()->route('enquiry-qualities.index')
            ->with('success','Enquiry Quality updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $data = EnquiryQuality::find($request->id);

        if ($data->default == '1') {
            return response()->json([
                'success' => false,
                'message' => ['Default record can not be deleted.']
            ]);
        }

        $data->delete();

        $Redirect = 'enquiry-qualities';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = EnquiryQuality::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Enquiry Quality status successfully change'],
                    'data'=>[
                       'redirect'=>'/users/',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change status'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }

    public function changedefault(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('default' => $request->status );
            $count  = EnquiryQuality::where(['default' => $request->status])->count();

            if ($count > 0 && $request->status == 1) {
                return response()->json([
                    'success' => false,
                    'message' => ['There is already a default Enquiry Quality!'],
                    'data'    => []
                ]);
            }

            $Update = EnquiryQuality::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success' => true,
                    'message' => ['Enquiry Quality default status successfully changed.']
                ]);
            } else {
                return response()->json([
                   'success' => false,
                   'message' => ['Error for change default']
                ]);
            }
        }
    }
}
