<?php

namespace App\Http\Controllers\Enquiries;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\EnquiryType;

class EnquiryTypesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:enquiry-types-list', ['only' => ['index']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $data = EnquiryType::select([
                    'enquiry_types.id',
                    'enquiry_types.enquiry_type_name',
                    'enquiry_types.enquiry_type_priority',
                    'enquiry_types.default',
                    'enquiry_types.status',
                    'companies.name as company'
                ])
                ->leftjoin('companies', 'companies.id', 'enquiry_types.company_id');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                            return $query->where('enquiry_types.enquiry_type_name', 'like', '%'.$search_query.'%')
                                ->orWhere('enquiry_types.enquiry_type_details', 'like', '%'.$search_query.'%')
                                ->orWhere('enquiry_types.additional3', 'like', '%'.$search_query.'%');
                        })
                        ->paginate($sort_by);

            return view('enquiry-types.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('enquiry-types.index',compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $priority = EnquiryType::max('enquiry_type_priority');
        $priority = $priority != '' ? $priority : 0;

        return view('enquiry-types.create',compact('priority'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'enquiry_type_name'     => 'required|unique:enquiry_types',
            'enquiry_type_priority' => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $data['company_id'] = \Auth::user()->company_id;

        $default_check = EnquiryType::where(['default' => 1,'status' => 1])->first();

        if (!$default_check) {
            $data['default'] = 1;
        }

        $category = EnquiryType::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Enquiry Type created successfully',
                'data'    => $category
            ]);
        }

        return redirect()->route('enquiry-types.index')
            ->with('success','Enquiry Type created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $type = EnquiryType::find($id);

        return view('enquiry-types.edit',compact('type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'enquiry_type_name'     => 'required|unique:enquiry_types,enquiry_type_name,'.$id,
            'enquiry_type_priority' => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $type = EnquiryType::find($id);

        if (!$type) {
            return redirect()->route('enquiry-types.index')
                ->with('error','Enquiry Type not found!');
        }

        $type->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Enquiry Type updated successfully',
            ]);
        }

        return redirect()->route('enquiry-types.index')
            ->with('success','Enquiry Type updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $data = EnquiryType::find($request->id);

        if ($data->default == '1') {
            return response()->json([
                'success' => false,
                'message' => ['Default record can not be deleted.']
            ]);
        }

        $data->delete();

        $Redirect = 'enquiry-types';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = EnquiryType::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Enquiry Type status successfully change'],
                    'data'=>[
                       'redirect'=>'/users/',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change status'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }

    public function changedefault(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('default' => $request->status );
            $count  = EnquiryType::where(['default' => $request->status])->count();

            if ($count > 0 && $request->status == 1) {
                return response()->json([
                    'success' => false,
                    'message' => ['There is already a default Enquiry Type!'],
                    'data'    => []
                ]);
            }

            $Update = EnquiryType::where('id', '=', $request->id)->update($data);

            if($Update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Enquiry Type default status successfully changed.'],
                    'data'=>[
                       'redirect'=>'/users/',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change default'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }
}
