<?php

namespace App\Http\Controllers\MedicalSetting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\FindingCategory;

class FindingCategoriesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:finding-categories-list', ['only' => ['index']]);
    }

    public function index(Request $request)
    {
        $data = FindingCategory::select([
                    'finding_categories.id',
                    'finding_categories.category',
                    'finding_categories.status',
                    'finding_categories.priority',
                ])
                ->orderBy('priority', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('finding_categories.category', 'like', '%'.$search_query.'%');
            })
            ->paginate($sort_by);

            return view('finding-categories.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('finding-categories.index', compact('data'));
        }
    }

    public function create()
    {
    	$priority = FindingCategory::max('priority')+1;

        return view('finding-categories.create', compact('priority'));
    }

    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'category' => 'required|unique:finding_categories,category,NULL,id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $priority = FindingCategory::max('priority')+1;
        $data['priority'] = $priority;

        $finding_category = FindingCategory::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Finding Category created successfully.',
                'data'    => $finding_category
            ]);
        }
    }

    public function edit($id)
    {
        $data = FindingCategory::find($id);

        return view('finding-categories.edit', compact('data'));
    }

    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'category' => 'required|unique:finding_categories,category,'.$id.',id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $finding_category = FindingCategory::find($id);

        if (!$finding_category) {
            return redirect()->back()
                ->with('error','Finding Category not found!');
        }

        $finding_category->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Finding Category Updated successfully',
                'data'    => $finding_category
            ]);
        }
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = FindingCategory::where('id', '=', $request->id)->update($data);

            $Redirect = 'test-units';

            if($Update){
                return response()->json([
                    'success' => true,
                    'message' => ['Finding Category status successfully change'],
                    'data'    => [
                        'redirect' => $Redirect,
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => ['Error for change status'],
                    'data'    => [
                        'redirect' => '',
                    ]
                ]);
            }
        }
    }

    public function destroy(Request $request)
    {
        FindingCategory::find($request->id)->delete();

        $Redirect = 'symptoms-classifications';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
