<?php

namespace App\Http\Controllers\MedicalSetting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\OperationCategory;

class OperationCategoriesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:operation-categories-list', ['only' => ['index']]);
    }

    public function index(Request $request)
    {
        $data = OperationCategory::select([
                    'operation_categories.id',
                    'operation_categories.category',
                    'operation_categories.status',
                    'operation_categories.priority',
                ])
                ->orderBy('priority', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('operation_categories.category', 'like', '%'.$search_query.'%');
            })
            ->paginate($sort_by);

            return view('operation-categories.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('operation-categories.index', compact('data'));
        }
    }

    public function create()
    {
    	$priority = OperationCategory::max('priority')+1;

        return view('operation-categories.create', compact('priority'));
    }

    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'category' => 'required|unique:operation_categories,category,NULL,id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $priority = OperationCategory::max('priority')+1;
        $data['priority'] = $priority;

        $operation_category = OperationCategory::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Operation Category created successfully.',
                'data'    => $operation_category
            ]);
        }
    }

    public function edit($id)
    {
        $data = OperationCategory::find($id);

        return view('operation-categories.edit', compact('data'));
    }

    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'category' => 'required|unique:operation_categories,category,'.$id.',id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $operation_category = OperationCategory::find($id);

        if (!$operation_category) {
            return redirect()->back()
                ->with('error','Operation Category not found!');
        }

        $operation_category->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Operation Category Updated successfully',
                'data'    => $operation_category
            ]);
        }
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = OperationCategory::where('id', '=', $request->id)->update($data);

            $Redirect = 'test-units';

            if($Update){
                return response()->json([
                    'success' => true,
                    'message' => ['Operation Category status successfully change'],
                    'data'    => [
                        'redirect' => $Redirect,
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => ['Error for change status'],
                    'data'    => [
                        'redirect' => '',
                    ]
                ]);
            }
        }
    }

    public function destroy(Request $request)
    {
        OperationCategory::find($request->id)->delete();

        $Redirect = 'symptoms-classifications';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
