<?php

namespace App\Http\Controllers\MedicalSetting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SymptomClassification;

class SymptomClassificationsController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:symptom-classifications-list', ['only' => ['index']]);
    }

    public function index(Request $request)
    {
        $data = SymptomClassification::select([
                    'symptom_classifications.id',
                    'symptom_classifications.code',
                    'symptom_classifications.symptoms_type as symptoms_type',
                    'symptom_classifications.status',
                    'symptom_classifications.priority',
                ])
                ->orderBy('priority', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('symptom_classifications.symptoms_type', 'like', '%'.$search_query.'%');
            })
            ->paginate($sort_by);

            return view('symptom-classifications.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('symptom-classifications.index', compact('data'));
        }
    }

    public function create()
    {
    	$priority = SymptomClassification::max('priority')+1;

        return view('symptom-classifications.create', compact('priority'));
    }

    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'symptoms_type' => 'required|unique:symptom_classifications,symptoms_type,NULL,id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $priority = SymptomClassification::max('priority')+1;
        $data['priority'] = $priority;

        $symptom_classification = SymptomClassification::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Symptom Classification created successfully.',
                'data'    => $symptom_classification
            ]);
        }
    }

    public function edit($id)
    {
        $data = SymptomClassification::find($id);

        return view('symptom-classifications.edit', compact('data'));
    }

    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'symptoms_type' => 'required|unique:symptom_classifications,symptoms_type,'.$id.',id',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $symptom_classification = SymptomClassification::find($id);

        if (!$symptom_classification) {
            return redirect()->back()
                ->with('error','Symptom Classification not found!');
        }

        $symptom_classification->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Symptom Classification Updated successfully',
                'data'    => $symptom_classification
            ]);
        }
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = SymptomClassification::where('id', '=', $request->id)->update($data);

            $Redirect = 'test-units';

            if($Update){
                return response()->json([
                    'success' => true,
                    'message' => ['Symptom Classification status successfully change'],
                    'data'    => [
                        'redirect' => $Redirect,
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => ['Error for change status'],
                    'data'    => [
                        'redirect' => '',
                    ]
                ]);
            }
        }
    }

    public function destroy(Request $request)
    {
        SymptomClassification::find($request->id)->delete();

        $Redirect = 'symptoms-classifications';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }
}
