<?php

namespace App\Http\Controllers\TaskManager;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TaskComment;

class CommentsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $data = TaskComment::leftJoin('users as tagged_user' , 'task_comments.user_id' , '=' , 'tagged_user.id')
            ->leftJoin('users' , 'task_comments.created_by' , '=' , 'users.id')
            ->select(
                'task_comments.id',
                'task_comments.comment',
                'task_comments.created_at',
                'users.name',
                'tagged_user.name as tagged_user',
            )->orderBy('task_comments.created_at', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('task_comments.comment', 'like', '%'.$search_query.'%')
                ->orWhere('users.name', 'like', '%'.$search_query.'%');
            })->paginate($sort_by);

            return view('task-manager.comments.table', compact('data'));
        } else {
            $data = $data->paginate(10);

            return view('task-manager.comments.table', compact('data'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'comment'       => 'required',
            'id'            => 'required',
        ]);

        if ($validator->fails()) {

            return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                'redirect' => 'pipeline-stages.index'],
            ]);
        }

        $pipeline_stages = TaskComment::create([
            'task_id'    => $request->id,
            'comment'    => $request->comment,
            'user_id'    => $request->tagged_user,
            'created_by' => \Auth()->id(),
        ]);

        return response()->json([
            'success'  => true,
            'message'  => 'Comment Created successfully',
            'data'     => [
            ]
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data = TaskComment::leftJoin('users as tagged_user' , 'task_comments.user_id' , '=' , 'tagged_user.id')
            ->select(
                'task_comments.id',
                'task_comments.comment',
                'task_comments.user_id',
                'tagged_user.name as tagged_user',
            )
            ->where('task_comments.id', $id)
            ->first();

        if (!$data) {
            return response()->json(['status' => 'error', 'data' => [], 'message' => 'Record not found!']);
        }

        return response()->json(['status' => 'success', 'data' => $data]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'comment'       => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                'redirect' => 'pipeline-stages.index'],
            ]);
        }
        $data = TaskComment::find($id);

        if (!$data) {
            return response()->json(['success' => false, 'data' => [], 'message' => 'Record not found!']);
        }

        $data->update([
            'comment'    => $request->comment,
            'user_id'    => $request->tagged_user
        ]);

        return response()->json([
            'success'  => true,
            'message'  => 'Comment updated successfully',
            'data'     => [
            ]
        ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $id = $request->id ?? '';
        $data = TaskComment::find($id);

        if (!$data) {
            return response()->json(['success' => false, 'data' => [], 'message' => 'Record not found!']);
        }

        $data->delete();

        return response()->json([
            'success'  => true,
            'message'  => 'Comment destroyed successfully',
            'data'     => [
            ]
        ]);
    }
}
