<?php

namespace App\Http\Controllers\TaskManager;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TaskGroup;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;

class TaskGroupsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $task_groups = TaskGroup::select(
            'task_groups.id as task_groups_id',
            'task_groups.name as name',
            'task_groups.description as description',
            'task_groups.created_by as created_by',
            'task_groups.status as status',
            'task_groups.priority as priority',
        )->orderBy('task_groups.name', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $task_groups = $task_groups->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('task_groups.name', 'like', '%'.$search_query.'%')
                ->orWhere('task_groups.description', 'like', '%'.$search_query.'%');
            })->paginate($sort_by);

            return view('task-manager.task-groups.table', compact('task_groups'));
        } else {
            $task_groups = $task_groups->paginate(10);

            return view('task-manager.task-groups.index', compact('task_groups'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function createTaskGroup()
    {
       // $task_groups = TaskGroup::select(
       //      'task_groups.name as name',
       //      'task_groups.description as description',
       //  )->get();

        return view('accounts.create-task-groups');
    } 

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeTaskGroup(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'name'           => 'required',
            // 'description' => 'required',
        ]);

        if ($validator->fails()) {

            return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                    'redirect' => 'task-groups.index',
                ]
            ]);
        }

        $task_groups = TaskGroup::create([
            'name'        => $request->name,
            'description' => $request->description,
            'created_by'  => Auth::user()->id ,
        ]);

        return response()->json([
            'success'  => true,
            'message'  => 'Task Group Created successfully',
            'data'     => [
                'redirect' => 'task-groups.index', 
            ]
        ]);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $task_groups = TaskGroup::findOrFail($id);

        return view('accounts.edit-task-groups', compact('task_groups'));

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateTaskGroup(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'name'        => 'required',
            // 'description' => 'required',
        ]);

        if ($validator->fails()) {
           return response()->json([
                'success'  => false,
                'message'  => $validator->getMessageBag()->first(),
                'data'     => [
                    'redirect' => 'task-groups.index'],
            ]);
        }

        $task_groups = TaskGroup::findOrFail($id);
        $task_groups->update($request->all());

        return response()->json([
            'success'  => true,
            'message'  => 'Task Group updated successfully',
            'data'     => [
                'redirect' => 'task-groups.index', 
            ]
        ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $task_groups = TaskGroup::findOrFail($id);
        $task_groups->delete();

        return response()->json([
            'success' => true,
            'message' => 'Task Group Deleted successfully',
            'data' => [
                'redirect' => '', 
            ]
        ]);
    }


    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $update = TaskGroup::where('id', '=', $request->id)->update($data);

            if($update){
                return response()->json([
                    'success'=>true,
                    'message'=>['Task Group status successfully change'],
                    'data'=>[
                       'redirect'=>'',
                       'reload'=>true,
                    ]
                ]);
            } else {
                return response()->json([
                   'success'=>false,
                   'message'=>['Error for change status'],
                   'data'=>[
                       'redirect'=>'',
                   ]
                ]);
            }
        }
    }


}