<?php

namespace App\Http\Controllers\TestManager;

use App\Http\Controllers\Controller;
use App\Models\SampleCollectionTube;
use App\Models\TestMethod;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class SampleCollectionTubesController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:collection-tube-list', ['only' => ['index']]);
    }

    public function index(Request $request)
    {
        $data = SampleCollectionTube::select([
                    'sample_collection_tubes.id',
                    'sample_collection_tubes.tube_name',
                    'sample_collection_tubes.code',
                    'sample_collection_tubes.details',
                    'sample_collection_tubes.color_name',
                    'sample_collection_tubes.color_code',
                    'sample_collection_tubes.anticoagulant',
                    'sample_collection_tubes.text_name',
                    'sample_collection_tubes.inversion_to_mix',
                    'sample_collection_tubes.status',
                ])
                ->orderBy('sample_collection_tubes.tube_name', 'ASC');

        if ($request->ajax()) {
            $sort_by      = $request->get('sortby') ?? 10;
            $sort_type    = $request->get('sorttype');
            $search_query = $request->get('query');

            $data = $data->when(!empty($search_query), function ($query) use ($search_query) {
                return $query->where('sample_collection_tubes.tube_name', 'like', '%'.$search_query.'%')
                    ->orWhere('sample_collection_tubes.color_name', 'like', '%'.$search_query.'%')
                    ->orWhere('sample_collection_tubes.color_code', 'like', '%'.$search_query.'%')
                    ->orWhere('sample_collection_tubes.anticoagulant', 'like', '%'.$search_query.'%')
                    ->orWhere('sample_collection_tubes.text_name', 'like', '%'.$search_query.'%')
                    ->orWhere('sample_collection_tubes.inversion_to_mix', 'like', '%'.$search_query.'%');
            })
            ->paginate($sort_by);

            return view('sample-collection-tubes.table', compact('data'));
        }
        else
        {
            $data = $data->paginate(10);

            return view('sample-collection-tubes.index', compact('data'));
        }
    }

    public function create()
    {
        return view('sample-collection-tubes.create');
    }

    public function store(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'tube_name'        => 'required|unique:sample_collection_tubes,tube_name,NULL,id,deleted_at,NULL',
            'color_name'       => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $tube = SampleCollectionTube::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Sample Collection Tube created successfully.',
                'data'    => $tube
            ]);
        }
    }

    public function edit($id)
    {
        $data = SampleCollectionTube::find($id);

        return view('sample-collection-tubes.edit',compact('data'));
    }

    public function update(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'tube_name'        => 'required|unique:sample_collection_tubes,tube_name,'.$id.',id,deleted_at,NULL',
            'color_name'       => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->getMessageBag()->first(),
                    'data'    => []
                ]);
            }

            return redirect()->back()->with('error', $validator->getMessageBag()->first());
        }

        $data = $request->except([
            '_token',
            '_method'
        ]);

        $tube = SampleCollectionTube::find($id);

        if (!$tube) {
            return redirect()->back()
                ->with('error','Sample Collection Tube not found!');
        }

        $tube->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Sample Collection Tube Updated successfully',
                'data'    => $tube
            ]);
        }
    }

    public function destroy(Request $request)
    {
        SampleCollectionTube::find($request->id)->delete();

        $Redirect = 'sample-collection-tubes';

        return response()->json([
            'success' => true,
            'message' => ['Deleted successfully'],
            'data'    => [
                'redirect' => $Redirect,
            ]
        ]);
    }

    public function changeStatus(Request $request)
    {
        if ($request->ajax()) {
            $data   = array('status' => $request->status );
            $Update = SampleCollectionTube::where('id', '=', $request->id)->update($data);

            $Redirect = 'sample-collection-tubes';

            if($Update){
                return response()->json([
                    'success' => true,
                    'message' => ['Sample Collection Tube status successfully change'],
                    'data'    => [
                        'redirect' => $Redirect,
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => ['Error for change status'],
                    'data'    => [
                        'redirect' => '',
                    ]
                ]);
            }
        }
    }
}
